﻿namespace Microsoft.Samples.PlanMyNight.Infrastructure.Mvc
{
    using System;
    using System.Collections.Generic;
    using System.Globalization;
    using System.Linq;
    using System.Text;
    using System.Web.Mvc;
    using System.Web.UI;

    public static class AjaxClientScriptExtensions
    {
        private interface IScriptInclude
        {
            string Render(bool useCdn);
        }

        public static bool JavaScriptEnabled(this AjaxHelper ajaxHelper)
        {
            return System.Web.HttpContext.Current.Request.Browser.EcmaScriptVersion.Major >= 1;
        }

        public static void RegisterClientScriptInclude(this AjaxHelper ajaxHelper, string url)
        {
            RegisterClientScriptInclude(ajaxHelper, url, null);
        }

        public static void RegisterClientScriptInclude(this AjaxHelper ajaxHelper, string url, string cdnUrl)
        {
            var scripts = GetScriptReferences(ajaxHelper);
            if (scripts.OfType<ScriptIncludeReference>().FirstOrDefault(s => s.Url == url) == null)
            {
                scripts.Add(new ScriptIncludeReference(url, cdnUrl));
            }
        }

        public static void RegisterCombinedScriptInclude(this AjaxHelper ajaxHelper, string url)
        {
            RegisterCombinedScriptInclude(ajaxHelper, url, null);
        }

        public static void RegisterCombinedScriptInclude(this AjaxHelper ajaxHelper, string url, string groupName)
        {
            var scripts = GetScriptReferences(ajaxHelper);

            var combinedScript = scripts.OfType<CombinedScriptIncludeReference>().FirstOrDefault(s => string.Equals(s.Group, groupName, StringComparison.OrdinalIgnoreCase));
            if (combinedScript == null)
            {
                var appPath = ajaxHelper.ViewContext.HttpContext.Request.ApplicationPath.TrimEnd('/');
                combinedScript = new CombinedScriptIncludeReference(appPath)
                {
                    Group = groupName
                };
                scripts.Add(combinedScript);
            }

            if (!combinedScript.Urls.Contains(url))
            {
                combinedScript.Urls.Add(url);
            }
        }

        public static void RegisterClientScriptBlock(this AjaxHelper ajaxHelper, string script)
        {
            GetScriptReferences(ajaxHelper).Add(new ScriptIncludeBlock(script));
        }

        public static string RenderClientScripts(this AjaxHelper ajaxHelper)
        {
            var useCdnSettingValue = System.Configuration.ConfigurationManager.AppSettings["useCdn"];

            var useCdn = true;

            if (!string.IsNullOrEmpty(useCdnSettingValue))
            {
                useCdn = Convert.ToBoolean(useCdnSettingValue);
            }

            return RenderClientScripts(ajaxHelper, useCdn);
        }

        public static string RenderClientScripts(this AjaxHelper ajaxHelper, bool useCdn)
        {
            StringBuilder sb = new StringBuilder();
            foreach (IScriptInclude reference in GetScriptReferences(ajaxHelper))
            {
                sb.Append(reference.Render(useCdn));
            }

            GetScriptReferences(ajaxHelper).Clear();

            return sb.ToString();
        }

        private static List<IScriptInclude> GetScriptReferences(AjaxHelper ajaxHelper)
        {
            var contextItems = ajaxHelper.ViewContext.HttpContext.Items;
            if (!contextItems.Contains("AjaxClientScripts"))
            {
                contextItems["AjaxClientScripts"] = new List<IScriptInclude>();
            }

            return (List<IScriptInclude>)contextItems["AjaxClientScripts"];
        }

        private class ScriptIncludeReference : IScriptInclude
        {
            public ScriptIncludeReference(string url, string cdnUrl)
            {
                this.Url = url;
                this.CdnUrl = cdnUrl;
            }

            public string Url { get; private set; }

            public string CdnUrl { get; private set; }

            public string Render(bool useCdn)
            {
                return string.Format(
                    CultureInfo.InvariantCulture,
                    "<script type=\"text/javascript\" src=\"{0}\"></script>",
                    useCdn && this.CdnUrl != null ? this.CdnUrl : this.Url);
            }
        }

        private class CombinedScriptIncludeReference : IScriptInclude
        {
            private readonly string baseUrl;

            public CombinedScriptIncludeReference(string baseUrl)
            {
                this.baseUrl = baseUrl;
                this.Urls = new List<string>();
            }

            public ICollection<string> Urls
            {
                get;
                private set;
            }

            public string Group
            {
                get;
                set;
            }

            public string Render(bool useCdn)
            {
                return string.Format(
                    CultureInfo.InvariantCulture,
                    "<script type=\"text/javascript\" src=\"{0}/JsCombine.ashx?js={1}\"></script>",
                    this.baseUrl,
                    string.Join(",", this.Urls.Select(u => System.Web.HttpUtility.UrlEncodeUnicode(u)).ToArray()));
            }
        }

        private class ScriptIncludeBlock : IScriptInclude
        {
            public ScriptIncludeBlock(string script)
            {
                this.Script = script;
            }

            public string Script { get; private set; }

            public string Render(bool useCdn)
            {
                return string.Format(
                    CultureInfo.InvariantCulture,
@"<script type=""text/javascript"">
//<![CDATA[
{0}
//]]>
</script>",
                    this.Script);
            }
        }
    }
}
